﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace MicroRWD.UI.MFIC
{
    public partial class MicroRWD_KeysPanel : UserControl
    {
        #region Private Properties
        private const int C_COUNT = 6;
        private bool formloaded = false;
        private byte keyNumber = 0x00;
        private const byte C_LAST_KEY = 0x1F;        
        private bool updateKeyFields = true;
        private bool updateDataFields = true;
        private byte[] data = new byte[C_COUNT];
        public byte[] Data
        {
            get { return data; }
            set { data = value; }
        }

        #endregion

        #region Constructors

        public MicroRWD_KeysPanel()
        {
            InitializeComponent();
        }

        #endregion

        #region Private Methods

        private void refreshView()
        {
            keyNumTextBox.Text = keyNumber.ToString();

            // Update Data fields
            if (updateDataFields)
            {
                // Save the current text selection
                int selectionStart = hexTextBox.SelectionStart;
                int selctionLength = hexTextBox.SelectionLength;

                hexTextBox.Text = String.Join(string.Empty, Array.ConvertAll(Data, b => b.ToString("X2") + ' '));

                hexTextBox.SelectionStart = selectionStart;
                hexTextBox.SelectionLength = selctionLength;

                // Save the current text selection
                selectionStart = asciiTextBox.SelectionStart;
                selctionLength = asciiTextBox.SelectionLength;

                asciiTextBox.Text = String.Join(string.Empty, Array.ConvertAll(Data, b => ((b > 31) && (b < 127)) ? ((char)b).ToString() + ' ' : ". "));

                asciiTextBox.SelectionStart = selectionStart;
                asciiTextBox.SelectionLength = selctionLength;
            }
        }

        private void validateValues()
        {
            // Ensure the RWD Key Number is within its range
            if (keyNumber < 0x00)
            {
                keyNumber = 0x00;
            }
            else if (keyNumber > C_LAST_KEY)
            {
                keyNumber = C_LAST_KEY;
            }
        }

        private void updateTextboxKeyCodes(object sender, KeyEventArgs e, ref byte _value, ref bool _updateFlag)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;
            if (e.KeyCode == Keys.Delete)
            {
                if (textbox.SelectionLength > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues();
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
            else if (e.KeyCode == Keys.Back)
            {
                if (textbox.SelectionStart > 0)
                {
                    _value = 0;
                    textbox.SelectionStart = 0;
                    textbox.SelectionLength = 1;

                    // Refresh view
                    validateValues();
                    _updateFlag = true;
                    refreshView();
                }
                e.Handled = true;
            }
        }

        private void updateDecTextKeyChars(object sender, KeyPressEventArgs e, ref byte _value, ref bool _updateFlag, bool _isStartAddr)
        {
            System.Windows.Forms.TextBox textbox = (System.Windows.Forms.TextBox)sender;

            // Ignore non dec digit key presses
            if (IsDecDigit(e.KeyChar))
            {
                // Is cursor in front of a digit                    
                // replace digit
                // move cursor to next position (select digit if one is there)
                int cur = textbox.SelectionStart;
                char[] curVal = textbox.Text.ToCharArray();
                if ((cur >= 0) && (cur <= 2))
                {
                    if (cur > (curVal.Length - 1))
                    {
                        char[] temp = new char[curVal.Length + 1];
                        Array.Copy(curVal, 0, temp, 0, curVal.Length);
                        temp[curVal.Length] = e.KeyChar;

                        try
                        {
                            _value = byte.Parse(new string(temp));
                        }
                        catch (OverflowException)
                        {
                            _value = byte.MaxValue;
                        }
                    }
                    else
                    {
                        curVal[cur] = e.KeyChar;
                        _value = byte.Parse(new string(curVal));
                    }

                    ++cur;
                    textbox.SelectionStart = Math.Min(cur, textbox.MaxLength);
                    textbox.SelectionLength = (textbox.SelectionStart < 2) ? 1 : 0;
                    _updateFlag = true;
                }

                // Refresh view
                validateValues();
                refreshView();
            }
            e.Handled = true;
        }

        private bool IsHexDigit(char _c)
        {
            if (((_c >= '0') && (_c <= '9')) || ((_c >= 'A') && (_c <= 'F')) || ((_c >= 'a') && (_c <= 'f')))
            {
                return true;
            }

            return false;
        }

        private bool IsDecDigit(char _c)
        {
            if ((_c >= '0') && (_c <= '9'))
            {
                return true;
            }

            return false;
        }

        private byte HexToByte(char _c)
        {
            byte result = 0;

            if ((_c >= '0') && (_c <= '9'))
            {
                result = (byte)(_c - '0');
            }
            else if ((_c >= 'A') && (_c <= 'F'))
            {
                result = (byte)(10 + (_c - 'A'));
            }
            else if ((_c >= 'a') && (_c <= 'f'))
            {
                result = (byte)(10 + (_c - 'a'));
            }

            return result;
        }

        #endregion

        #region Event Handlers

        private void MicroRWD_KeysPanel_Load(object sender, EventArgs e)
        {
            formloaded = true;
        }

        private void keyNumNextButton_Click(object sender, EventArgs e)
        {
            // Increment the key number
            if (keyNumber < C_LAST_KEY)
            {
                ++keyNumber;
                validateValues();
                refreshView();
            }
        }

        private void keyNumPrevButton_Click(object sender, EventArgs e)
        {
            // Decrement the key number
            if (keyNumber > 0x00)
            {
                --keyNumber;
                validateValues();
                refreshView();
            }
        }

        private void keyNumTextBox_KeyDown(object sender, KeyEventArgs e)
        {
            updateTextboxKeyCodes(sender, e, ref keyNumber, ref updateKeyFields);
        }

        private void keyNumTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            updateDecTextKeyChars(sender, e, ref keyNumber, ref updateKeyFields, false);
        }

        private void hexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (C_COUNT > 0)
            {
                // Process backspace and hex digit key presses
                if (e.KeyChar == (char)8)
                {
                    // XX_XX_XX_XX_XX_XX
                    hexTextBox.SelectionStart = (hexTextBox.SelectionStart > 0) ? hexTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    refreshView();
                }
                else if (IsHexDigit(e.KeyChar))
                {
                    // XX_XX_XX_XX_XX_XX
                    int cur = hexTextBox.SelectionStart;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }
                    int pos = Math.Min(cur / 3, C_COUNT - 1);

                    if (Data == null)
                    {
                        Data = new byte[C_COUNT];
                    }

                    if ((cur % 3) == 0)
                    {
                        // hi byte
                        Data[pos] = (byte)(((HexToByte(e.KeyChar) << 4) & 0xf0) | (Data[pos] & 0x0f));
                    }
                    else
                    {
                        // lo byte
                        Data[pos] = (byte)((Data[pos] & 0xf0) | (HexToByte(e.KeyChar) & 0x0f));
                    }

                    // XX_XX_XX_XX_XX_XX
                    ++cur;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }

                    hexTextBox.SelectionStart = Math.Min(cur, hexTextBox.MaxLength - 1);

                    // Refresh view
                    updateDataFields = true;
                    refreshView();
                }
            }

            e.Handled = true;
        }

        private void asciiTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (C_COUNT > 0)
            {
                // Process backspace and ascii key presses
                if (e.KeyChar == (char)8)
                {
                    // ._._._.
                    asciiTextBox.SelectionStart = (asciiTextBox.SelectionStart > 0) ? asciiTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    refreshView();
                }
                else
                {
                    // ._._._._._.
                    int cur = asciiTextBox.SelectionStart;
                    if ((cur % 1) == 1)
                    {
                        ++cur;
                    }
                    int pos = Math.Min((cur + 1) / 2, C_COUNT - 1);

                    if (Data == null)
                    {
                        Data = new byte[C_COUNT];
                        updateDataFields = true;
                    }

                    Data[pos] = (byte)e.KeyChar;

                    cur += 2;

                    asciiTextBox.SelectionStart = Math.Min(cur, asciiTextBox.MaxLength - 1);
                    asciiTextBox.SelectionLength = 1;

                    // Refresh view
                    refreshView();
                }
            }

            e.Handled = true;
        }

        private void writeButton_Click(object sender, EventArgs e)
        {
            string dataStr = String.Join(string.Empty, Array.ConvertAll(Data, b => b.ToString("X2") + ' '));
            DialogResult result = MessageBox.Show("Write data " + dataStr + " to RWD Key number " + keyNumber.ToString() + "?", "Confirm Write Key", MessageBoxButtons.OKCancel);
            if(result == DialogResult.OK)
            {
                Program.Controller.Reader.CmdKeyStore(keyNumber, Data);
            }
        }

        #endregion
    }
}
